/*
 * Query the FMS for current status
 */
#include <time.h>
#include <stdio.h>
#include <unistd.h>

#include "libfma.h"
#include "lf_fms.h"

void show_fabric_status(int fms);

static void
usage()
{
  fprintf(stderr, "Queries the fms and prints a status summary.\n\n");
  fprintf(stderr, "Usage: fm_status\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -s <fms_server> - specifies IP address of fms node\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

int
main(
  int argc,
  char **argv)
{
  int c;
  char *fms_names;
  int fms;
  extern char *optarg;

  lf_init();

  /* defaults */
  fms_names = NULL;

  /*
   * Get command line args
   */
  while ((c = getopt(argc, argv, "hs:V")) != EOF) switch (c) {
  case 'h':
    usage();
    break;
  case 's':
    fms_names = optarg;
    break;
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;
  }

  /*
   * Open connection to fms
   */
  fms = lf_fms_connect(fms_names);
  if (fms == -1) {
    fprintf(stderr, "Error connecting to ");
    if (fms_names == NULL) {
      fprintf(stderr, "default FMS\n");
    } else {
      fprintf(stderr, "FMS in (%s)\n", fms_names);
    }
    exit(1);
  }

  /*
   * Read and display status
   */
  show_fabric_status(fms);

  /* all done, shut down connection */
  lf_fms_disconnect(fms);

  exit(0);
}

/*
 * Query the FMS for current status
 */
void
show_fabric_status(
  int fms)
{
  struct lf_fms_status status;
  int rc;

  /* get the list of alerts from the FMS */
  rc = lf_fms_query_status(fms, &status);
  if (rc != 0) {
    fprintf(stderr, "Error reading status from FMS\n");
    exit(1);
  }

  /* Print the results of the status query */
  printf("FMS Fabric status\n\n");
  printf("%d\thosts known\n", status.hosts_known);
  printf("%d\tFMAs found\n", status.fmas_found);
  printf("%d\tun-ACKed alerts\n", status.unacked_alerts);

  if (status.mapping_complete) {
    printf("Mapping is complete, last map generated by %s\n",
	status.last_mapper[0] != '\0' ? status.last_mapper : "<none>");
  } else {
    printf("Mapping in progress, mapper is %s\n", status.last_mapper);
  }

  if (status.db_complete) {
    printf("Database is complete\n");
  } else {
    printf("Database generation not yet complete.\n");
  }
}
